#  ----------------------------------------------------------------------------
#          ATMEL Microcontroller Software Support  -  ROUSSET  -
#  ----------------------------------------------------------------------------
#  Copyright (c) 2006, Atmel Corporation
#
#  All rights reserved.
#
#  Redistribution and use in source and binary forms, with or without
#  modification, are permitted provided that the following conditions are met:
#
#  - Redistributions of source code must retain the above copyright notice,
#  this list of conditions and the disclaiimer below.
#
#  - Redistributions in binary form must reproduce the above copyright notice,
#  this list of conditions and the disclaimer below in the documentation and/or
#  other materials provided with the distribution. 
#
#  Atmel's name may not be used to endorse or promote products derived from
#  this software without specific prior written permission. 
#
#  DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
#  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
#  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
#  DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
#  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
#  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
#  OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
#  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
#  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
#  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#  ----------------------------------------------------------------------------
#############################################################################################################################
#############################################################################################################################
#  <GROUP Files>
#
#  Module:
#    $Source: /srv/cvs/cvs.system20/4NETOS/BIN/SAM-BA/Updated\040scripts/ATMEL\040Corporation/AT91-ISP\040v1.10/SAM-BA\040v2.6/lib/ROBO-Interface2008/SDRAM.tcl,v $
#  Version:
#    $Revision: 1.1.2.3 $
#  History:
#    $Log: SDRAM.tcl,v $
#    Revision 1.1.2.3  2008/09/29 13:11:58  bo
#    ROBO_IF08
#    - bugfix tcl path  syntax
#
#    Revision 1.1.2.2  2008/09/18 13:15:15  bo
#    ROBO_IF08
#    - Board "ROBO-Interface2008": SDRAM setup
#
#
#  Description:
#    SDRAM initialization function for the <Board ROBO-Interface2008>
#
#  Copyright:
#    2008 MSC Vertriebs GmbH, Smart Network Devices, Germany. All Rights Reserved.
#   All copyright and other intellectual property rights in this software are and
#   shall remain the property of MSC Vertriebs GmbH, Germany. Unauthorized
#   reproduction or distribution of this program, or any portion of it, may result in
#   severe civil and criminal penalties and will be prosecuted to the maximum extent
#   possible under law.
#
#############################################################################################################################
#############################################################################################################################

################################################################################
################################################################################
## NAMESPACE SDRAM
################################################################################
################################################################################
namespace eval SDRAM {
    variable   SDRAM_initialized  [expr 0]
}

################################################################################
#  proc SDRAM::initSDRAM_already_up
################################################################################
proc SDRAM::initSDRAM_already_up {} {
          
    variable SDRAM_initialized

    set SDRAM_initialized 1
    puts "-I- set SDRAM to <already up>"
}


################################################################################
#  proc SDRAM::initSDRAM
################################################################################
#  100 MHz
#  16 bit
#  SDRAM 4/8M x16
#  4M x16
#  K4S641632K-T(U)UC75
#  ds_k4s64xx32k_rev11.pdf
################################################################################
proc SDRAM::initSDRAM {} {

    global AT91C_CCFG_EBICSA AT91C_MATRIX_CS1A
    global AT91C_SDRAMC_NC_8
    global AT91C_SDRAMC_NR_12
    global AT91C_SDRAMC_NB_4_BANKS
    global AT91C_SDRAMC_CAS_2
    global AT91C_SDRAMC_DBW_16_BITS
    global AT91C_SDRAMC_TWR_1
    global AT91C_SDRAMC_TRC_5
    global AT91C_SDRAMC_TRP_2
    global AT91C_SDRAMC_TRCD_2
    global AT91C_SDRAMC_TRAS_2
    global AT91C_SDRAMC_TXSR_4
    global AT91C_SDRAMC_CR
    global AT91C_EBI_SDRAM
    global AT91C_SDRAMC_MODE_PRCGALL_CMD AT91C_SDRAMC_MODE_RFSH_CMD AT91C_SDRAMC_MODE_LMR_CMD AT91C_SDRAMC_MODE_NORMAL_CMD AT91C_SDRAMC_MR
    global AT91C_SDRAMC_TR

    global target
    set dummy_err 0

    variable SDRAM_initialized
    if {$SDRAM_initialized != 0} {
        puts "-I- SDRAM already initialized"
        return
    }

    puts "-I- Init MATRIX to support EBI0 CS1 for SDRAM"
    set register_value [expr [TCL_Read_Int $target(handle) $AT91C_CCFG_EBICSA dummy_err] | $AT91C_MATRIX_CS1A]
    TCL_Write_Int $target(handle) $register_value $AT91C_CCFG_EBICSA dummy_err

    puts "-I- Init SDRAM geometry, timing"
    puts "-I- 1. A minimum pause of 200us is provided to precede any signal toggle"
    set register_value 0x00000000
    set register_value [expr $register_value | $AT91C_SDRAMC_NC_8           ]
    set register_value [expr $register_value | $AT91C_SDRAMC_NR_12          ]
    set register_value [expr $register_value | $AT91C_SDRAMC_NB_4_BANKS     ]
    set register_value [expr $register_value | $AT91C_SDRAMC_CAS_2          ]
    set register_value [expr $register_value | $AT91C_SDRAMC_DBW_16_BITS    ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TWR_1          ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TRC_5          ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TRP_2          ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TRCD_2         ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TRAS_2         ]
    set register_value [expr $register_value | $AT91C_SDRAMC_TXSR_4         ]
    TCL_Write_Int $target(handle) $register_value $AT91C_SDRAMC_CR dummy_err

    puts "-I- 2. A Precharge All command is issued to the SDRAM"
    TCL_Write_Int $target(handle) $AT91C_SDRAMC_MODE_PRCGALL_CMD $AT91C_SDRAMC_MR dummy_err

    puts "-I- SDRAM = 0;"
    TCL_Write_Int $target(handle) 0 $AT91C_EBI_SDRAM dummy_err

    puts "-I- 3. Eight Auto-refresh are provided"
    TCL_Write_Int $target(handle) $AT91C_SDRAMC_MODE_RFSH_CMD $AT91C_SDRAMC_MR dummy_err
    
    for {set i 0} {$i < 8} {incr i} {
        puts "-I- SDRAM = 0;"
        TCL_Write_Int $target(handle) 0 $AT91C_EBI_SDRAM dummy_err
    }

    puts "-I- 4. A mode register cycle is issued to program the SDRAM parameters"
    TCL_Write_Int $target(handle) $AT91C_SDRAMC_MODE_LMR_CMD $AT91C_SDRAMC_MR dummy_err

    puts "-I- SDRAM = 0;"
    TCL_Write_Int $target(handle) 0 [expr $AT91C_EBI_SDRAM] dummy_err

    puts "-I- 5. Write refresh rate into SDRAMC refresh timer COUNT register"
    TCL_Write_Int $target(handle) 1562 $AT91C_SDRAMC_TR dummy_err

    puts "-I- 6. A Normal Mode Command is provided, 3 clocks after tMRD is set"
    TCL_Write_Int $target(handle) $AT91C_SDRAMC_MODE_NORMAL_CMD $AT91C_SDRAMC_MR dummy_err

    puts "-I- SDRAM = 0;"
    TCL_Write_Int $target(handle) 0 $AT91C_EBI_SDRAM dummy_err

    set SDRAM_initialized 1
    puts "-I- End of Init_SDRAM"
}


################################################################################
#  proc SDRAM::sendFile
################################################################################
proc SDRAM::sendFile { name addr } {
    
    global valueOfDataForSendFile
    global target
    set dummy_err 0

    variable SDRAM_initialized
    if {$SDRAM_initialized != 1} {
        puts stderr "-E- SDRAM not initialized"
        return
    }

    if {[catch {set f [open $name r]}]} {
        set valueOfDataForSendFile 0
        puts stderr "-E- Can't open file $name"
        return -1
    }
    
    fconfigure $f -translation binary
    
    set size [file size $name]
    puts "-I- File size = $size byte(s)"
    set valueOfDataForSendFile [read $f $size]
    
    close $f

    if {[catch {TCL_Write_Data $target(handle) $addr valueOfDataForSendFile $size dummy_err}]} {
        puts stderr "-E- Can't send data, error in connection"
        set valueOfDataForSendFile 0
        return
    }
    
    set valueOfDataForSendFile 0
}


################################################################################
#  proc SDRAM::receiveFile
################################################################################
proc SDRAM::receiveFile {name addr size} {

    variable SDRAM_initialized
    if {$SDRAM_initialized != 1} {
        puts stderr "-E- SDRAM not initialized"
        return
    }

    global target
    set dummy_err 0

    #read data from target
    if {[catch {set result [TCL_Read_Data $target(handle) $addr $size dummy_err]}]} {
        puts stderr "-E- Can't read data, error in connection"
        return -1
    }

    # put data in a file
    if {[catch {set f2 [open $name w+]}]} {
        puts stderr "-E- Can't open file $name"
        return -1
    }

    fconfigure $f2 -translation binary
    puts -nonewline $f2 $result
    close $f2
}

